*==============================================================================
* Program:			SFUpdate.prg
* Purpose:			Check for and install updates
* Author:			Doug Hennig
* Last revision:	08/25/2014
* Parameters:		tcEXE             - the path to the EXE to restart
*					tcRegistryKey     - the Registry key to use for settings
*					tcUpdatesURL      - the FTP URL to download updates from
*					tcUserName        - the user name for the FTP server
*					tcPassword        - the password for the FTP server
*					tcUpdateFile      - the name of the file containing update
*						information on the FTP server
*					tlDisplayChecking - .T. to display a "checking for
*						updates" dialog 
* Returns:			none
* Environment in:	none
* Environment out:	the application may have been updated
*==============================================================================

lparameters tcEXE, ;
	tcRegistryKey, ;
	tcUpdatesURL, ;
	tcUserName, ;
	tcPassword, ;
	tcUpdateFile, ;
	tlDisplayChecking
local loRegistry as 'SFRegistry' OF 'SFRegistry.vcx', ;
	loUpdate as 'SFUpdate' OF 'SFUpdate.prg', ;
	loForm as Form
local lcKey, ;
	llCheck, ;
	lnDays, ;
	ldLastDate, ;
	laVersion[1], ;
	lnResult, ;
	llVisible, ;
	llUpdate

* If we're not displaying a "checking" dialog (likely because we're being
* called at startup rather than from a menu item), see when the last time we
* checked for an update was.

loRegistry = newobject('SFRegistry', 'SFRegistry.vcx')
lcKey      = addbs(tcRegistryKey) + 'Updates'
if tlDisplayChecking
	llCheck = .T.
else
	lnDays     = val(loRegistry.GetKey(lcKey, 'UpdateDays', '1'))
	ldLastDate = ctod(loRegistry.GetKey(lcKey, 'LastUpdate', ''))
	llCheck    = empty(ldLastDate) or ldLastDate + lnDays <= date()
endif tlDisplayChecking

* If we're due for a check, do so.

if llCheck

* Instantiate an SFUpdate object and set the settings it needs.

	loUpdate = newobject('SFUpdate', 'Updater.prg')
	loUpdate.cUpdatesURL = tcUpdatesURL
	loUpdate.cUserName   = tcUserName
	loUpdate.cPassword   = tcPassword
	loUpdate.cUpdateFile = tcUpdateFile
	loUpdate.cEXEPath    = tcEXE

* Get the current version number for the specified EXE.

	agetfileversion(laVersion, tcEXE)
	loUpdate.cCurrVersion = laVersion[4]

* Display a dialog about looking for an updated version.

	if tlDisplayChecking
		do form AccessingServer with 'Checking for updates.', ;
			'Connecting to server', 'Getting update information' ;
			name loForm linked
		doevents force
		bindevent(loUpdate, 'Connected',      loForm, 'Connected', 1)
		bindevent(loUpdate, 'HaveUpdateFile', loForm, 'GotResult', 1)
	endif tlDisplayChecking

* Check for an update.

	lnResult = loUpdate.CheckUpdate()

* Wait a couple of seconds, then close the dialog.

	if tlDisplayChecking
		declare Sleep in Win32API integer nMilliseconds
		Sleep(2000)
		loForm.Release()
	endif tlDisplayChecking
	do case

* An update is available so ask the user if they want to download it.

		case lnResult = 0
			if not _screen.Visible and type('_screen.ActiveForm') <> 'O'
				llVisible       = .F.
				_screen.Icon    = 'MyApp.ico'
				_screen.Left    = -30000
				_screen.Visible = .T.
			endif not _screen.Visible ...
			do form DownloadPrompt with loUpdate.cUpdateText name loForm linked
			if vartype(loForm) = 'O'
				loForm.Release()
				llUpdate = .T.
			endif vartype(loForm) = 'O'
			if not llVisible
				_screen.Visible = .F.
			endif not llVisible

* Get the update if we're supposed to.

			if llUpdate
				do form Progress with 'Downloading updates.', ;
					'The latest version of this application is being ' + ;
					'downloaded. This may take a few minutes.', ;
					'download.png' name loForm linked
				loUpdate.oProgress = loForm
				loUpdate.Update()
				loForm.Release()
			endif llUpdate

* A problem occurred downloading the update.

		case lnResult = 1
			messagebox('The application could not be updated at this time. ' + ;
				'Please try again later.', 16, 'Application Updater')

* An update is available but the current version is less than the minimum
* version supported for automatic updating.

		case lnResult = 2
			messagebox('A new version is available but it cannot be ' + ;
				'automatically installed. Please download and install the ' + ;
				'new version.', 16, 'Application Updater')

* No update is available or we already have it.

		case lnResult = 3
			messagebox('You are running the most recent version of this ' + ;
				'application.', 64, 'Application Updater')
	endcase
endif llCheck

* Save the current date as the last date checked.

loRegistry.SetKey(lcKey, 'LastUpdate', date())


* This is here to avoid a compile error in wwFTP.prg.

function CharToBin
endfunc
