* Update class.

define class SFUpdate as Custom

* Define constants for control characters.

	#define ccCR	chr(13)
	#define ccLF	chr(10)
	#define ccCRLF	chr(13) + chr(10)
	#define ccTAB	chr(9)

* Define the properties clients can set or read.

	cUpdatesURL      = ''
		&& the URL to download updates from
	cUserName        = ''
		&& the FTP user name
	cPassword        = ''
		&& the FTP password
	cEXEPath           = ''
		&& the path to the main EXE for the application
	cUpdateFile      = 'files.xml'
		&& the name of the file containing information about any update
	cCurrVersion     = ''
		&& the current application version
	cTargetDirectory = sys(2023)
		&& the folder to write files to (defaults to user's temp folder)
	cUpdateText      = ''
		&& information about the update
	cErrorMessage    = ''
		&& the text of any error that occurred
	oProgress        = .NULL.
		&& a progress dialog that shows the download progress
	cTitle           = 'Application Updater'
		&& the title of a UAC warning message

* Protected properties.

	protected cServerName
		&& the FTP server name
	protected cServerDirectory
		&& the FTP folder name
	protected cVersion
		&& the update version
	protected cMinVersion
		&& the minimum application version the update can handle
	protected aFiles[1]
		&& a list of files to download
	protected oFTP
		&& an FTP object

* See if an update is available.

	function CheckUpdate()
		local lnPos, ;
			llReturn, ;
			llUpdateAvailable, ;
			llUpdate, ;
			lnResult
		with This

* Determine the server name and directory.

			lnPos = at('/', .cUpdatesURL)
			if lnPos = 0
				lnPos = at('\', .cUpdatesURL)
			endif lnPos = 0
			.cServerName      = left(.cUpdatesURL, lnPos - 1)
			.cServerDirectory = substr(.cUpdatesURL, lnPos + 1)
			if right(.cServerDirectory, 1) <> '/'
				.cServerDirectory = .cServerDirectory + '/'
			endif right(.cServerDirectory, 1) <> '/'
			.cTargetDirectory = addbs(.cTargetDirectory)

* Retrieve the update file.

			llReturn = .GetUpdateFile()
			do case

* If we received the update file list, raise the HaveUpdateFile event and see
* if the server version number is newer than the current one and the current
* version is greater than or equal to the minimum version (if it's specified).

				case llReturn
					raiseevent(This, 'HaveUpdateFile')
					llUpdateAvailable = not empty(.cVersion) and ;
						not empty(.cCurrVersion) and ;
						.cVersion > .cCurrVersion
					llUpdate = llUpdateAvailable and ;
						(empty(.cMinVersion) or ;
						.cCurrVersion >= .cMinVersion)
					do case

* An update is available.

						case llUpdate
							lnResult = 0

* An update is available but the current EXE is less than the minimum version.

						case llUpdateAvailable
							lnResult = 2

* We're running the latest version.

						otherwise
							lnResult = 3
					endcase

* We couldn't receive the update file list because something went wrong.

				case not empty(.cErrorMessage)
					lnResult = 1

* The update file list didn't exist, so assume we're running the latest
* version.

				otherwise
					lnResult = 3
			endcase
		endwith
		return lnResult
	endfunc

* Download and install the update.

	function Update()
		local loProcess as 'API_AppRun' OF 'API_AppRun.PRG'
		local lnFiles, ;
			lcFiles, ;
			lnI, ;
			llUpdate, ;
			llReturn, ;
			lcFile, ;
			lcDirectory, ;
			lcBat, ;
			lcUpdateFile, ;
			lcUpdateApp
		with This

* Go through each file and if there's a min or max version for it, figure out
* which files we'll use; remove the ones we won't from aFiles.

			lnFiles = alen(.aFiles, 1)
			lcFiles = ''
			for lnI = lnFiles to 1 step -1
				if (not empty(.aFiles[lnI, 2]) and ;
						.cCurrVersion < .aFiles[lnI, 2]) or ;
					(not empty(.aFiles[lnI, 3]) and ;
						.cCurrVersion > .aFiles[lnI, 3])
					lnFiles = lnFiles - 1
					if lnFiles = 0
						.aFiles  = ''
						llUpdate = .F.
					else
						adel(.aFiles, lnI)
						dimension .aFiles[lnFiles, 3]
					endif lnFiles = 0
				else
					lcFiles = lcFiles + ;
						forcepath(.aFiles[lnI, 1], .cTargetDirectory) + ;
						ccCR
				endif (not empty(.aFiles[lnI, 2]) ...
			next lnI

* Get the update.

			llReturn   = .DownloadFiles()
			lcFile     = .cTargetDirectory + upper(.aFiles[1, 1])
			lcDirectory = addbs(curdir())
		endwith
		do case

* If we successfully downloaded the update files and there's a single file
* named SETUP.EXE or UPDATE.EXE (or some variant like SETUP12345.EXE), create a
* batch file to run it and ourselves afterward. Tell the user they'll get a UAC
* prompt in Vista or later.

			case llReturn and lnFiles = 1 and ;
				('SETUP' $ lcFile or 'UPDATE' $ lcFile) and ;
				justext(lcFile) = 'EXE'
				text to lcBat pretext 1 + 2 noshow textmerge
					"<<lcFile>>" /silent /dir="<<lcDirectory>>"
					del "<<lcFile>>"
					"<<This.cEXEPath>>"
				endtext
				lcUpdateApp = This.cTargetDirectory + 'Update.bat'
				strtofile(lcBat, lcUpdateApp)
				if os(3) >= '6'
					This.DisplayUACWarning(lcFile)
				endif os(3) >= '6'
				loProcess = newobject('API_AppRun', 'API_AppRun.PRG', '', ;
					lcUpdateApp, '', 'HID')
				loProcess.LaunchApp()
				on shutdown
				quit

* If we successfully downloaded the update files, we have to run an external
* program to do the update since presumably it contains a newer version of this
* EXE.

			case llReturn
				lcUpdateFile = forcepath(sys(2015) + '.txt', ;
					This.cTargetDirectory)
				strtofile(lcFiles, lcUpdateFile)
				lcUpdateApp = forcepath('UpdateApp.exe', lcDirectory)

* Tell the user they'll get a UAC prompt in Vista or later, then run UpdateApp.

				if os(3) >= '6'
					This.DisplayUACWarning(lcUpdateApp)
				endif os(3) >= '6'
				declare integer ShellExecute in SHELL32.DLL ;
					integer nWinHandle, ;
					string cOperation, ;
					string cFileName, ;
					string cParameters, ;
					string cDirectory, ;
					integer nShowWindow
				ShellExecute(0, 'RunAs', lcUpdateApp, ;
					'"' + _screen.Caption + '" "' + lcUpdateFile + ;
					'" "' + This.cEXEPath + '"', lcDirectory, 1)
				on shutdown
				quit
		endcase
		return llReturn
	endfunc

* Display a warning about UAC.

	function DisplayUACWarning(tcFile)
		local lcMessage
		lcMessage = 'Windows User Access Control will prompt you to run ' + ;
			justfname(tcFile) + '. Please choose Allow so it can update ' + ;
			'the program files.'
		messagebox(lcMessage, 64, This.cTitle)
	endfunc

* Download the update file and process its contents.

	protected function GetUpdateFile
		local lcFile, ;
			llReturn, ;
			lcXML, ;
			lcFiles, ;
			lnFiles, ;
			lnI
		with This

* If a path wasn't specified for the update file, use cTargetDirectory.

			lcFile = .cUpdateFile
			if empty(justpath(lcFile))
				lcFile = forcepath(lcFile, .cTargetDirectory)
			endif empty(justpath(lcFile))

* Download the update file.

			llReturn = .GetFile(lcFile)

* If we succeeded, get the version information and list of files to download.

			if llReturn and file(lcFile)
				lcXML = filetostr(lcFile)
				erase (lcFile)
				.cUpdateText = .HTMLDecode(strextract(lcXML, '<text>', '</text>'))
				.cVersion    = strextract(lcXML, '<version>',    '</version>')
				.cMinVersion = strextract(lcXML, '<minversion>', '</minversion>')
				lcFiles      = strextract(lcXML, '<files>',      '</files>')
				lnFiles      = occurs('<file', lcFiles)
				if lnFiles > 0
					dimension .aFiles[lnFiles, 3]
					for lnI = 1 to lnFiles
						lcFile = strextract(lcFiles, '<file', '</file>', lnI)
						.aFiles[lnI, 1] = substr(lcFile, at('>', lcFile) + 1)
						.aFiles[lnI, 2] = strextract(lcFile, 'minversion="', '"')
						.aFiles[lnI, 3] = strextract(lcFile, 'maxversion="', '"')
					next lnI
				endif lnFiles > 0
			endif llReturn ...
		endwith
		return llReturn
	endfunc

* Gets the specified file from the FTP site.

	protected function GetFile(tcFile)
		local lcFile, ;
			lnResult, ;
			llReturn
		with This

* If we already have an FTP object, use it. If not, create one and connect to
* the FTP site.

			if vartype(.oFTP) = 'O' or .Connect()

* If a path wasn't specified, use cTargetDirectory.

				lcFile = tcFile
				if empty(justpath(lcFile))
					lcFile = forcepath(lcFile, .cTargetDirectory)
				endif empty(justpath(lcFile))

* Erase any existing file and then download it.

				try
					erase (lcFile)
				catch
				endtry
				lnResult = .oFTP.FTPGetFileEx(.cServerDirectory + justfname(tcFile), ;
					lcFile)
				llReturn = lnResult = 0
				do case

* If the file doesn't exist, blank cErrorMessage to signal that nothing went
* wrong.


					case not llReturn and .oFTP.nError = 12003
						.cErrorMessage = ''

* If something went wrong, set cErrorMessage accordingly.

					case not llReturn
						.cErrorMessage = .cErrorMessage + ;
							iif(empty(.cErrorMessage), '', ccCR) + ;
							'GetFile: ' + .oFTP.cErrorMsg

* If the download was canceled, we'll return .F.

					case .oFTP.lCancelDownload
						llReturn = .F.
				endcase
			endif vartype(.oFTP) = 'O' ...
		endwith
		return llReturn
	endfunc

* Connect to the FTP site using wwFTP.

	protected function Connect
		local llReturn, ;
			lnResult, ;
			loException as Exception
		with This
			.oFTP = newobject('wwFTP', 'wwFTP.prg')
			.oFTP.cUserName       = .cUserName
			.oFTP.cPassword       = .cPassword
			.oFTP.nConnectTimeout = 10
			.oFTP.lPassiveFTP     = .T.
				&& required for Windows Server 2008 and later
			try
				lnResult = .oFTP.FTPConnect(.cServerName)
				llReturn = lnResult = 0
			catch to loException
			endtry
			if llReturn
				bindevent(.oFTP, 'OnFTPBufferUpdate', This, 'UpdateProgress')
			else
				.cErrorMessage = .cErrorMessage + ;
					iif(empty(.cErrorMessage), '', ccCR) + ;
					'Connect: ' + iif(empty(.oFTP.cErrorMsg), ;
					'Cannot connect to update site', .oFTP.cErrorMsg)
			endif llReturn
		endwith

* Raise the Connected event if we connected.

		if llReturn
			raiseevent(This, 'Connected')
		endif llReturn
		return llReturn
	endfunc

* Update the progress dialog as files are downloaded.

	protected function UpdateProgress(tnBytesDownloaded, tnBufferReads, ;
		tcCurrentChunk, tnTotalBytes, toFtp)
		if vartype(This.oProgress) = 'O'
			This.oProgress.Update(tnBytesDownloaded, tnTotalBytes)
			if This.oProgress.lCancel
				toFTP.lCancelDownload = .T.
			endif This.oProgress.lCancel
		endif vartype(This.oProgress) = 'O'
	endfunc

* Download the specified files.

	protected function DownloadFiles
		local lnFiles, ;
			lnI, ;
			lcFile, ;
			llReturn
		with This
			lnFiles = alen(.aFiles, 1)
			for lnI = 1 to lnFiles
				lcFile = .aFiles[lnI, 1]
				if not empty(lcFile)
					llReturn = .GetFile(lcFile)
					if not llReturn
						exit
					endif not llReturn
				endif not empty(lcFile)
			next lnI
		endwith
		return llReturn
	endfunc

* Decode HTML encoded text.

	protected function HTMLDecode(tcString)
		local lcString
		lcString = strtran(tcString, '&amp;',  '&')
		lcString = strtran(lcString, '&gt;',   '>')
		lcString = strtran(lcString, '&lt;',   '<')
		lcString = strtran(lcString, '&quot;', '"')
		lcString = strtran(lcString, '&apos;', "'")
		lcString = strtran(lcString, '&#39;',  "'")
		lcString = strtran(lcString, '&#47;',  '/')
		lcString = strtran(lcString, '&#13;',  ccCR)
		lcString = strtran(lcString, '&#10;',  ccLF)
		lcString = strtran(lcString, '&#9;',   ccTAB)
		return lcString
	endfunc

* The HaveUpdateFile event.

	function HaveUpdateFile
	endfunc

* The Connected event.

	function Connected
	endfunc
enddefine
