
************************************************************************************
** Merge Class for Merging Plain Text, RTF, HTML, or other Text Documents
** Default Delimiters are [[ and ]]
** You can change This.StartDelim and This.EndDelim if you want different delimiters
** There are several Merge functions depending on what you need to do
************************************************************************************
DEFINE CLASS Merge as Custom 
  #DEFINE CASE_INSENSITIVE 1
  #DEFINE FIRST_OCCUR 1
  #DEFINE OVERWRITE_FILE 0
  #DEFINE APPEND_FILE 1
  #DEFINE READ_WRITE_BUFFERED 2
  #DEFINE SEEK_FROM_EOF 2
  #DEFINE SW_SHOWNORMAL    1
  #DEFINE SW_HIDE 0
  
  StartDelim = '[['
  EndDelim = ']]'


  HIDDEN FUNCTION Init
    DECLARE INTEGER ShellExecute ;
	    IN SHELL32.DLL ;
	    INTEGER nWinHandle,;
	    STRING cOperation,;   
	    STRING cFileName,;
	    STRING cParameters,;
	    STRING cDirectory,;
	    INTEGER nShowWindow
  ENDFUNC 


  ********************************************************************
  ** This is the central function 
  ** The other Merge.... functions call this one to do the work
  ********************************************************************
  FUNCTION MergeStringToString(tcTemplateContent as String) as String
    LOCAL lcVar, lcValue, lcResult
    lcResult = tcTemplateContent
    
    ** First merge variable, if any
    lcVar = STREXTRACT(lcResult, This.StartDelim, This.EndDelim, FIRST_OCCUR, CASE_INSENSITIVE)

	** Replace Merge Variables until there are no more to be found
    DO WHILE NOT EMPTY(lcVar)      
	  lcValue = This.SafeEval(lcVar)      
      ** replace the merge var and delimiters with the value we just evaluated
      lcResult = StrTran(lcResult, ;
                         This.StartDelim + lcVar + This.EndDelim, ;
                         lcValue)
      ** get the next merge var for next loop around                   
      lcVar = STREXTRACT(lcResult, This.StartDelim, This.EndDelim, FIRST_OCCUR, CASE_INSENSITIVE)
	ENDDO 
    RETURN lcResult    
  ENDFUNC 


  FUNCTION MergeFileToString(tcTemplateFile as String) as String
    Return This.MergeStringToString(FILETOSTR(tcTemplateFile))
  ENDFUNC 


  ************************************************************************
  ** Build up an output file by merging in another file
  ** usually adding one page at a time

  ** RTF documents start with "{" and end with "}"
  ** To add RTF pages together to create a full document
  ** you need to remove the "{" at the beginning and the "}" at the end
  ** of each page
  ************************************************************************
  FUNCTION MergeFileToFile(tcTemplateFile as String, tcResultFile as String, tnFileMode as Integer) as VOID
    ASSERT PCOUNT() >= 3 MESSAGE "Must have 3 parameters"
    LOCAL lcPage, lnHandle, lnFileSize
    lcPage = This.MergeFileToString(tcTemplateFile)
    TRY 
      IF (tnFileMode = OVERWRITE_FILE) OR ( NOT FILE(tcResultFile))
        lnHandle = FCREATE(tcResultFile)
      ELSE
        lnHandle = FOPEN(tcResultFile, READ_WRITE_BUFFERED)
      ENDIF 
      
      lnFileSize = FSEEK(lnHandle,0,SEEK_FROM_EOF)   && Get File Size
      
      IF (lnFileSize > 0) AND (LEFT(lcPage,1) = '{') && if rtf append      
        =FSEEK(lnHandle,-1,SEEK_FROM_EOF)            && lose the last char '}' of existing doc - it signifies end of RTF doc
        =FWRITE(lnHandle,'\page'+SUBSTR(lcPage,2))   && lose first char '{' of new page - it signifies begin of RTF doc
      ELSE
        =FWRITE(lnHandle, lcPage)
      ENDIF 
    FINALLY
       =FCLOSE(lnHandle)
    ENDTRY
  ENDFUNC 


  **************************************************************************************************
  ** SafeEval:
  ** Parameter is string that is a FoxPro expression
  ** Evaluate() gives us the value
  ** we force result to string with TRANSFORM() and catch any errors 
  ** Version for VFP 8 uses Try / Catch
  ** Version for VFP 5,6,7 uses Error Trapping Object
  **************************************************************************************************
  #IF VAL(VERSION(4)) > 7.99  
	  HIDDEN FUNCTION SafeEval(tcVar as String)
	    ** Works with VFP 8
	      TRY 	 
	        ** attempt to eval this merge var
	        ** TRANSFORM will convert any type to a string
	        lcValue = ALLTRIM(TRANSFORM(EVALUATE(This.StripRTF(tcVar))))
	      CATCH 
	        ** in case merge var mis-spelled, etc.
	        lcValue = [***ERROR in ]+tcVar+[: ] + Message()+[***]
	      ENDTRY     
	      RETURN lcValue
	  ENDFUNC 
  #ELSE
	  HIDDEN FUNCTION SafeEval(tcvar as String)
	    ** Works with all versions of VFP
	    LOCAL lcEval
	    lcEval = CREATEOBJECT("tEvaluator")
	    RETURN lcEval.Eval(This.StripRTF(tcVar))
	  ENDFUNC 
  #ENDIF    


  ***********************************************************************
  ** Sometimes Word will insert stray formatting characters 
  ** between delimiters and a merge variable
  ** so we kill the unnecesary RTF formatting
  ***********************************************************************
  HIDDEN FUNCTION StripRTF(tcSource) As String
		LOCAL lnStart, lnEnd, lcResult, lnCurrentSpace
		lnCurrentSpace = 1
		lcResult = tcSource
		
		** Look for an opening curly brace
		lnStart = AT('{', lcResult)
		DO WHILE lnStart > 0
			** Find the first space following it,
			lnEnd = AT(' ', lcResult)
			DO WHILE lnEnd < lnStart
				lnCurrentSpace = lnCurrentSpace + 1
				lnEnd = AT(' ', lcResult, lnCurrentSpace)
			ENDDO		
			** Remove the characters
			lcResult = STUFF(lcResult, lnStart, lnEnd - lnStart + 1, "")
			
			** Get the next opening curly brace
			lnStart = AT('{', lcResult)
		ENDDO
		** Remove any ending curly braces
		lcResult = STRTRAN(lcResult, "}", "")
		
		RETURN lcResult
  ENDFUNC


  ***********************************************************************
  ** Launch Word, WordPad, WordPerfect, etc - let it do all the work
  ** For printing, Launch the default editor minimized and let it print
  ***********************************************************************
  FUNCTION Print(tcFileName as String) as VOID
     = ShellExecute(0,'print',tcFileName,'','',SW_HIDE)  
  ENDFUNC

  
  *********************************************************************************
  ** Launch Word, WordPad, WordPerfect, etc - let it do all the work
  ** For preview, Launch the default editor normal size and let the user see it
  *********************************************************************************
  FUNCTION Preview(tcFileName as String) as VOID
     = ShellExecute(0,'open',tcFileName,'','',SW_SHOWNORMAL)    
  ENDFUNC 

ENDDEFINE 


****************************************************************
**  Helper class to trap eval errors
**  Used for pre VFP8 SafeEval()
****************************************************************
Define Class tEvaluator as custom
	cRetVal  = ''

  	HIDDEN Function Error(nIndex, nError, cMethod, nLine)
	    This.cRetval = '***ERROR: ' + Message()+[***]
	EndFunc
	  
	Function Eval(tcVarName as String) as String
	  	LOCAL lcVarName
		lcVarName = tcVarName
	
		This.cRetval  = ALLTRIM(TRANSFORM(EVALUATE(lcVarName)))
		
	    Return This.cRetval
	ENDFUNC
ENDDEFINE 
